<?php
/**
 * HTTP requests processor - CURL (served by WP_HTTP)
 *
 * @author  YITH
 * @package YITH\StripeClient\Processors
 * @version 1.0.0
 */

namespace YITH\StripeClient\Processors;

use YITH\StripeClient\Interfaces\Processor;

defined( 'YITH_STRIPE_CLIENT_DIR' ) || exit;

if ( ! class_exists( 'YITH\StripeClient\Processors\Curl' ) ) {
	/**
	 * Stripe Payments API class
	 *
	 * @since 1.0.0
	 */
	class Curl implements Processor {

		/**
		 * Url of the remote server.
		 *
		 * @var string
		 */
		protected $url;

		/**
		 * Initialize connection with the remote server
		 *
		 * @param string $url Base url of the remote server.
		 *
		 * @throws \Exception When an error occurs with processor initialization.
		 */
		public function maybe_connect( $url ) {
			if ( ! class_exists( 'WP_Http' ) ) {
				throw new \Exception( 'You cannot use this processor outside of a WordPress environment' );
			}

			$url = esc_url( $url );

			if ( ! $url ) {
				throw new \Exception( 'Submitted invalid url for the remote server' );
			}

			$this->url = $url;
		}

		/**
		 * Calls an endpoint on the remote server
		 *
		 * @param string $method   HTTP method to use for the call.
		 * @param string $endpoint Endpoint to call on the server.
		 * @param array  $args     Optional array of arguments for the call. An example is listed below:
		 * [
		 *    'timeout'            => 30,
		 *    'reject_unsafe_urls' => true,
		 *    'blocking'           => true,
		 *    'sslverify'          => true,
		 *    'attempts'           => 0,
		 *    'headers'            => []
		 *    'body'               => ''
		 * ].
		 */
		public function call( $method, $endpoint, $args = array() ) {
			// generates basic connection params.
			 $full_url = "{$this->url}/{$endpoint}";
			$method    = strtoupper( $method );

			// build arguments array.
			$args = array_merge(
				array(
					'timeout'            => 5,
					'reject_unsafe_urls' => true,
					'blocking'           => true,
					'sslverify'          => true,
					'attempts'           => 0,
				),
				$args,
				array(
					'method' => $method,
				)
			);

			return wp_remote_request( $full_url, $args );
		}
	}
}
