<?php
/**
 * Intent model
 * Representation of Intent instance and operations that can be performed over it on remote server
 *
 * @author  YITH
 * @package YITH\StripeClient\Models
 * @version 1.0.0
 */

namespace YITH\StripeClient\Models;

use YITH\StripeClient\Client;
use YITH\StripeClient\Models\Abstracts\Model;
use YITH\StripeClient\Traits\Object_Read;
use YITH\StripeClient\Traits\Object_Create;
use YITH\StripeClient\Traits\Object_Update;

defined( 'YITH_STRIPE_CLIENT_DIR' ) || exit;

if ( ! class_exists( 'YITH\StripeClient\Models\Intent' ) ) {
	/**
	 * Representation of intent instance
	 *
	 * @since 1.0.0
	 * @property string                                                                      $id                          Unique identifier for the object.
	 * @property int                                                                         $amount                      Amount intended to be collected by this PaymentIntent. A positive integer representing how much to charge in the <a href="https://stripe.com/docs/currencies#zero-decimal">smallest currency unit</a> (e.g., 100 cents to charge $1.00 or 100 to charge ¥100, a zero-decimal currency). The minimum amount is $0.50 US or <a href="https://stripe.com/docs/currencies#minimum-and-maximum-charge-amounts">equivalent in charge currency</a>. The amount value supports up to eight digits (e.g., a value of 99999999 for a USD charge of $999,999.99).
	 * @property int                                                                         $amount_capturable           Amount that can be captured from this PaymentIntent.
	 * @property \Stripe\StripeObject                                                        $amount_details
	 * @property int                                                                         $amount_received             Amount that was collected by this PaymentIntent.
	 * @property null|string|\Stripe\StripeObject                                            $application                 ID of the Connect application that created the PaymentIntent.
	 * @property null|int                                                                    $application_fee_amount      The amount of the application fee (if any) that will be requested to be applied to the payment and transferred to the application owner's Stripe account. The amount of the application fee collected will be capped at the total payment amount. For more information, see the PaymentIntents <a href="https://stripe.com/docs/payments/connected-accounts">use case for connected accounts</a>.
	 * @property null|\Stripe\StripeObject                                                   $automatic_payment_methods   Settings to configure compatible payment methods from the <a href="https://dashboard.stripe.com/settings/payment_methods">Stripe Dashboard</a>
	 * @property null|int                                                                    $canceled_at                 Populated when <code>status</code> is <code>canceled</code>, this is the time at which the PaymentIntent was canceled. Measured in seconds since the Unix epoch.
	 * @property null|string                                                                 $cancellation_reason         Reason for cancellation of this PaymentIntent, either user-provided (<code>duplicate</code>, <code>fraudulent</code>, <code>requested_by_customer</code>, or <code>abandoned</code>) or generated by Stripe internally (<code>failed_invoice</code>, <code>void_invoice</code>, or <code>automatic</code>).
	 * @property string                                                                      $capture_method              Controls when the funds will be captured from the customer's account.
	 * @property null|string                                                                 $client_secret               <p>The client secret of this PaymentIntent. Used for client-side retrieval using a publishable key.</p><p>The client secret can be used to complete a payment from your frontend. It should not be stored, logged, or exposed to anyone other than the customer. Make sure that you have TLS enabled on any page that includes the client secret.</p><p>Refer to our docs to <a href="https://stripe.com/docs/payments/accept-a-payment?ui=elements">accept a payment</a> and learn about how <code>client_secret</code> should be handled.</p>
	 * @property string                                                                      $confirmation_method
	 * @property int                                                                         $created                     Time at which the object was created. Measured in seconds since the Unix epoch.
	 * @property string                                                                      $currency                    Three-letter <a href="https://www.iso.org/iso-4217-currency-codes.html">ISO currency code</a>, in lowercase. Must be a <a href="https://stripe.com/docs/currencies">supported currency</a>.
	 * @property null|string|\Stripe\Customer                                                $customer                    <p>ID of the Customer this PaymentIntent belongs to, if one exists.</p><p>Payment methods attached to other Customers cannot be used with this PaymentIntent.</p><p>If present in combination with <a href="https://stripe.com/docs/api#payment_intent_object-setup_future_usage">setup_future_usage</a>, this PaymentIntent's payment method will be attached to the Customer after the PaymentIntent has been confirmed and any required actions from the user are complete.</p>
	 * @property null|string                                                                 $description                 An arbitrary string attached to the object. Often useful for displaying to users.
	 * @property null|string|\Stripe\Invoice                                                 $invoice                     ID of the invoice that created this PaymentIntent, if it exists.
	 * @property null|\Stripe\StripeObject                                                   $last_payment_error          The payment error encountered in the previous PaymentIntent confirmation. It will be cleared if the PaymentIntent is later updated for any reason.
	 * @property null|string|\Stripe\Charge                                                  $latest_charge               The latest charge created by this payment intent.
	 * @property bool                                                                        $livemode                    Has the value <code>true</code> if the object exists in live mode or the value <code>false</code> if the object exists in test mode.
	 * @property \Stripe\StripeObject                                                        $metadata                    Set of <a href="https://stripe.com/docs/api/metadata">key-value pairs</a> that you can attach to an object. This can be useful for storing additional information about the object in a structured format. For more information, see the <a href="https://stripe.com/docs/payments/payment-intents/creating-payment-intents#storing-information-in-metadata">documentation</a>.
	 * @property null|\Stripe\StripeObject                                                   $next_action                 If present, this property tells you what actions you need to take in order for your customer to fulfill a payment using the provided source.
	 * @property null|string|\Stripe\Account                                                 $on_behalf_of                The account (if any) for which the funds of the PaymentIntent are intended. See the PaymentIntents <a href="https://stripe.com/docs/payments/connected-accounts">use case for connected accounts</a> for details.
	 * @property null|string|\Stripe\PaymentMethod                                           $payment_method              ID of the payment method used in this PaymentIntent.
	 * @property null|\Stripe\StripeObject                                                   $payment_method_options      Payment-method-specific configuration for this PaymentIntent.
	 * @property string[]                                                                    $payment_method_types        The list of payment method types (e.g. card) that this PaymentIntent is allowed to use.
	 * @property null|\Stripe\StripeObject                                                   $processing                  If present, this property tells you about the processing state of the payment.
	 * @property null|string                                                                 $receipt_email               Email address that the receipt for the resulting payment will be sent to. If <code>receipt_email</code> is specified for a payment in live mode, a receipt will be sent regardless of your <a href="https://dashboard.stripe.com/account/emails">email settings</a>.
	 * @property null|string|\Stripe\Review                                                  $review                      ID of the review associated with this PaymentIntent, if any.
	 * @property null|string                                                                 $setup_future_usage          <p>Indicates that you intend to make future payments with this PaymentIntent's payment method.</p><p>Providing this parameter will <a href="https://stripe.com/docs/payments/save-during-payment">attach the payment method</a> to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be <a href="https://stripe.com/docs/api/payment_methods/attach">attached</a> to a Customer after the transaction completes.</p><p>When processing card payments, Stripe also uses <code>setup_future_usage</code> to dynamically optimize your payment flow and comply with regional legislation and network rules, such as <a href="https://stripe.com/docs/strong-customer-authentication">SCA</a>.</p>
	 * @property null|\Stripe\StripeObject                                                   $shipping                    Shipping information for this PaymentIntent.
	 * @property null|string|\Stripe\Account|\Stripe\BankAccount|\Stripe\Card|\Stripe\Source $source                      This is a legacy field that will be removed in the future. It is the ID of the Source object that is associated with this PaymentIntent, if one was supplied.
	 * @property null|string                                                                 $statement_descriptor        For non-card charges, you can use this value as the complete description that appears on your customers’ statements. Must contain at least one letter, maximum 22 characters.
	 * @property null|string                                                                 $statement_descriptor_suffix Provides information about a card payment that customers see on their statements. Concatenated with the prefix (shortened descriptor) or statement descriptor that’s set on the account to form the complete statement descriptor. Maximum 22 characters for the concatenated descriptor.
	 * @property string                                                                      $status                      Status of this PaymentIntent, one of <code>requires_payment_method</code>, <code>requires_confirmation</code>, <code>requires_action</code>, <code>processing</code>, <code>requires_capture</code>, <code>canceled</code>, or <code>succeeded</code>. Read more about each PaymentIntent <a href="https://stripe.com/docs/payments/intents#intent-statuses">status</a>.
	 * @property null|\Stripe\StripeObject                                                   $transfer_data               The data with which to automatically create a Transfer when the payment is finalized. See the PaymentIntents <a href="https://stripe.com/docs/payments/connected-accounts">use case for connected accounts</a> for details.
	 * @property null|string                                                                 $transfer_group              A string that identifies the resulting payment as part of a group. See the PaymentIntents <a href="https://stripe.com/docs/payments/connected-accounts">use case for connected accounts</a> for details.
	 */
	class Intent extends Model {

		use Object_Read, Object_Create, Object_Update;

		/**
		 * Endpoint related to this object on remote server
		 *
		 * @var string
		 */
		protected static $endpoint = ':env/:brand/intent';

		/**
		 * Data structure {@see Model::$data_structure}
		 *
		 * @var array
		 */
		protected static $data_structure;

		/**
		 * Confirm data structure
		 *
		 * @var array
		 */
		protected static $confirm_data_structure;

		/**
		 * Returns data structure describing current object
		 *
		 * @return array Data structure ({@see self::$data_structure} for more info).
		 */
		public static function get_data_structure() {
			if ( ! self::$data_structure ) {
				self::$data_structure = array(
					'id'                        => array(
						'label'    => __( 'Intent id', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => null,
					),
					'amount'                    => array(
						'label'    => __( 'Intent amount', 'yith-stripe-client' ),
						'type'     => 'number',
						'required' => false,
						'default'  => 0,
					),
					'amount_capturable'         => array(
						'label'    => __( 'Capturable amount', 'yith-stripe-client' ),
						'type'     => 'number',
						'required' => false,
						'default'  => 0,
					),
					'automatic_payment_methods' => array(
						'enabled' => array(
							'label'    => __( 'Automatic methods flag', 'yith-stripe-client' ),
							'type'     => 'bool',
							'required' => false,
							'default'  => true,
						),
					),
					'capture_method'            => array(
						'label'    => __( 'Capture method', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => false,
					),
					'confirm'                   => array(
						'label'    => __( 'Confirm flag', 'yith-stripe-client' ),
						'type'     => 'bool',
						'required' => false,
						'default'  => false,
					),
					'client_secret'             => array(
						'label'    => __( 'Client secret', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => '',
					),
					'currency'                  => array(
						'label'    => __( 'Intent currency', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => '',
					),
					'customer'                  => array(
						'label'    => __( 'Customer', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => '',
					),
					'description'               => array(
						'label'    => __( 'Intent description', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => '',
					),
					'last_payment_error'        => array(
						'label'    => __( 'Last payment error', 'yith-stripe-client' ),
						'type'     => 'hash',
						'required' => false,
						'default'  => false,
					),
					'latest_charge'             => array(
						'label'    => __( 'Last charge', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => '',
					),
					'mandate_data'              => array(
						'label'    => __( 'Mandate', 'yith-stripe-client' ),
						'type'     => 'hash',
						'required' => false,
						'default'  => null,
					),
					'metadata'                  => array(
						'label'    => __( 'Intent metadata', 'yith-stripe-client' ),
						'type'     => 'hash',
						'required' => false,
						'default'  => false,
					),
					'next_action'               => array(
						'label'    => __( 'Next action details', 'yith-stripe-client' ),
						'type'     => 'hash',
						'required' => false,
						'default'  => false,
					),
					'payment_method'            => array(
						'label'    => __( 'Payment method', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => '',
					),
					'payment_method_types'      => array(
						'label'    => __( 'Payment method types', 'yith-stripe-client' ),
						'type'     => 'hash',
						'required' => false,
						'default'  => false,
					),
					'receipt_email'             => array(
						'label'    => __( 'Receipt email', 'yith-stripe-client' ),
						'type'     => 'email',
						'required' => false,
						'default'  => '',
					),
					'return_url'                => array(
						'label'      => __( 'Return url', 'yith-stripe-client' ),
						'type'       => 'text',
						'required'   => false,
						'default'    => null,
						'validation' => 'url',
					),
					'setup_future_usage'        => array(
						'label'    => __( 'Setup future usage flag', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => '',
					),
					'status'                    => array(
						'label'    => __( 'Status', 'yith-stripe-client' ),
						'type'     => 'text',
						'required' => false,
						'default'  => 'requires_confirmation',
					),
				);
			}

			return self::$data_structure;
		}

		/**
		 * Returns data structure used to confirm Intent
		 *
		 * @return array Data structure ({@see self::$confirm_data_structure} for more info).
		 */
		public static function get_confirm_data_structure() {
			if ( ! self::$confirm_data_structure ) {
				$data_structure    = self::get_data_structure();
				$confirm_structure = array();
				$keys              = array(
					'automatic_payment_methods',
					'return_url',
					'payment_method',
					'receipt_email',
					'setup_future_usage',
					'mandate_data',
				);

				foreach ( $keys as $structure_key ) {
					$confirm_structure[ $structure_key ] = $data_structure[ $structure_key ];
				}

				self::$confirm_data_structure = $confirm_structure;
			}

			return self::$confirm_data_structure;
		}

		/**
		 * Confirms current intent by sending required data.
		 *
		 * @param string $id   Intent id.
		 * @param array  $data Array of arguments.
		 * @return Intent Intent object.
		 * @throws \Exception When something goes wrong with the call to the server.
		 */
		public static function confirm( $id, $data ) {
			return self::get( Client::call( 'POST', self::get_endpoint( "$id/confirm" ), self::parse_data( $data, false, self::get_confirm_data_structure() ) ) );
		}

		/**
		 * Captures current intent by sending required data.
		 *
		 * @param string $id Intent id.
		 * @return Intent Intent object.
		 * @throws \Exception When something goes wrong with the call to the server.
		 */
		public static function capture( $id ) {
			return self::get( Client::call( 'POST', self::get_endpoint( "$id/capture" ) ) );
		}
	}
}
