<?php
/**
 * Apple Pay class
 *
 * @author  YITH <plugins@yithemes.com>
 * @package YITH PayPal Payments for WooCommerce
 * @version 1.0.0
 * @since 2.22.0
 */

defined( 'ABSPATH' ) || exit;
/**
 * Class YITH_PayPal_Apple_Pay
 */
class YITH_PayPal_Apple_Pay {

	/**
	 * Gateway environment
	 *
	 * @var string
	 */
	protected $environment = '';

	/**
	 * Enabled
	 *
	 * @var bool
	 */
	protected $enabled = false;

	/**
	 * Paypal Gateway Enabled
	 *
	 * @var bool
	 */
	protected $paypal_enabled = true;

	/**
	 * Is valid for country currency combination
	 *
	 * @var bool
	 */
	protected $is_valid = false;

	/**
	 * Payment Title
	 *
	 * @var string
	 */
	protected $payment_title = '';

	/**
	 * Settings
	 *
	 * @var array
	 */
	protected $settings = array();

	/**
	 * SDK url
	 *
	 * @var string
	 */
	protected static $sdk_url = 'https://applepay.cdn-apple.com/jsapi/v1/apple-pay-sdk.js';

	/**
	 * ALLOWED_USER_BROWSERS
	 *
	 * @var array
	 */
	public const ALLOWED_USER_BROWSERS = array( 'Safari' );
	/**
	 * ALLOWED_USER_DEVICES
	 *
	 * @var array
	 */
	public const ALLOWED_USER_DEVICES = array( 'Macintosh', 'iPhone', 'iPad', 'iPod' );

	/**
	 * Single instance of the class
	 *
	 * @since 1.0.0
	 * @var class YITH_PayPal_Apple_Pay
	 */
	protected static $instance;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Get settings.
		$this->payment_title  = __( 'Pay with Apple Pay', 'yith-paypal-payments-for-woocommerce' );
		$this->settings       = get_option( 'yith_ppwc_ap_gateway_options', array( 'enabled' => false ) );
		$this->is_valid       = yith_ppwc_check_currency_country_validity( $this->get_supported_country_currency(), wc_get_base_location()['country'], get_woocommerce_currency() );
		$this->enabled        = $this->is_enabled();
		$this->paypal_enabled = yith_ppwc_is_paypal_gateway_enabled();

		if ( $this->is_valid ) {
			add_filter( 'yith_ppwc_payment_methods_settings', array( $this, 'add_apay_payment_method_settings' ), 15 );
		}

		if ( $this->enabled && $this->paypal_enabled ) {
			add_action( 'wp_loaded', array( $this, 'load_domain_association_file' ) );
			/* hook apple pay button to the paypal buttons container */
			add_action( 'yith_ppwc_after_buttons', array( $this, 'add_button' ) );

			/* load scripts */
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts_style' ), 30 );
			add_filter( 'yith_ppwc_sdk_parameters', array( $this, 'add_apple_pay_to_sdk' ) );

			add_filter( 'yith_ppwc_customization_settings', array( $this, 'add_apay_style_settings' ), 15 );

			add_filter( 'yith_ppwc_method_title_icons', array( $this, 'add_icon_to_payment_method_title' ) );
		}
	}
	/**
	 * Add google pay icon to paypal payment method title
	 *
	 * @param array $icons list of icons.
	 * @return array
	 */
	public function add_icon_to_payment_method_title( $icons ) {
		array_push( $icons, 'applepay' );
		return $icons;
	}

	/**
	 * Manage the loading of the domain association file
	 *
	 * @param string $environment sandbox or live.
	 * @since 2.22.0
	 */
	public function load_domain_association_file( $environment = '' ) {
		if ( ! isset( $_SERVER['REQUEST_URI'] ) ) {
			return;
		}

		$env = empty( $environment ) ? YITH_PayPal::get_instance()->get_gateway()->get_environment() : $environment;

		$request_uri = (string) filter_var( wp_unslash( $_SERVER['REQUEST_URI'] ), FILTER_SANITIZE_URL );
		if ( strpos( $request_uri, '.well-known/apple-developer-merchantid-domain-association' ) !== false ) {
			$validation_string = 'sandbox' === $env ? $this->sandbox_validation_string() : $this->live_validation_string();
			nocache_headers();
			header( 'Content-Type: text/plain', true, 200 );
			echo $validation_string;// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			exit;
		}
	}

	/**
	 * Returns the sandbox validation string.
	 *
	 * @return string
	 */
	private function sandbox_validation_string(): string {
		return apply_filters(
			'yith_ppwc_applepay_sandbox_validation_string',
			'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'
		);
	}

	/**
	 * Returns the live validation string.
	 *
	 * @return string
	 */
	private function live_validation_string(): string {
		return apply_filters(
			'yith_ppwc__applepay_live_validation_string',
			'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'
		);
	}

	/**
	 * Check browser/device support
	 *
	 * @since 2.22.0
	 * @return boolean
	 */
	public function is_browser_supported() {
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$user_agent      = wp_unslash( $_SERVER['HTTP_USER_AGENT'] ?? '' );
		$browser_allowed = false;
		$device_allowed  = false;
		if ( $user_agent ) {
			foreach ( $this::ALLOWED_USER_BROWSERS as $allowed_browser ) {
				if ( strpos( $user_agent, $allowed_browser ) !== false ) {
					$browser_allowed = true;
					break;
				}
			}
			foreach ( $this::ALLOWED_USER_DEVICES as $allowed_device ) {
				if ( strpos( $user_agent, $allowed_device ) !== false ) {
					$device_allowed = true;
					break;
				}
			}
		}

		return $browser_allowed && $device_allowed;
	}

	/**
	 * Get apple pay capability status
	 *
	 * @since 2.22.0
	 * @return boolean
	 */
	public function can_apple_pay() {
		$can = false;
		$m   = YITH_PayPal_Merchant::get_merchant();
		$m->check_status();
		$caps = $m->get( 'capabilities' );
		if ( $caps ) {
			foreach ( $caps as $i => $values ) {
				if ( isset( $values['name'] ) && 'APPLE_PAY' === $values['name'] ) {
					$can = 'ACTIVE' === $caps[ $i ]['status'] ? true : false;
					break;
				}
			}
		}

		return $can;
	}

	/**
	 * Add Apple Pay button payment method to related tab settings
	 *
	 * @param array $settings array of options.
	 * @return array $settings list of settings.
	 */
	public function add_apay_payment_method_settings( $settings ) {
		$merchant = YITH_PayPal_Merchant::get_merchant();

		/* Apple Pay Options */
		$apay_activate_url       = $this->get_apple_pay_activation_url();
		$can_apple_pay           = $this->can_apple_pay() && $merchant->are_payments_receivable();
		$apay_option_description = '';
		$class                   = '';

		$apay_option_description = esc_html__( 'Enable to allow your customers to use Apple Pay to pay their orders', 'yith-paypal-payments-for-woocommerce' );

		if ( ! $can_apple_pay ) {
			$apay_option_description  = wp_kses_post( sprintf( _x( 'Apple Pay is not active in your Paypal account.', 'Admin description option', 'yith-paypal-payments-for-woocommerce' ), $apay_activate_url ) );
			$apay_option_description .= ' <a href="' . $apay_activate_url . '" target="_blank">' . esc_html_x( 'Please Activate it.', 'Admin description option', 'yith-paypal-payments-for-woocommerce' ) . '</a>';

			if ( 'not-active' === YITH_PayPal_Merchant::get_merchant()->is_active() ) {
				$apay_option_description = esc_html__( 'Please connect before you can enable this option.', 'yith-paypal-payments-for-woocommerce' );
			}

			if ( ! $merchant->are_payments_receivable() ) {
				// translators: placoholder is a link.
				$apay_option_description = sprintf( __( 'Payment no receivable. Please check <a href="%s">Onboarding Status</a>.', 'yith-paypal-payments-for-woocommerce' ), esc_url( admin_url( 'admin.php?page=yith_paypal_payments' ) ) );
			}
		}

		if ( ! $this->paypal_enabled ) {
			$apay_option_description = esc_html__( 'It seems Paypal gateway is disabled so you cannot enable Apple Pay', 'yith-paypal-payments-for-woocommerce' );
		}

		$ap_options = get_option( 'yith_ppwc_ap_gateway_options', array() );
		if ( ! $can_apple_pay || ! $this->paypal_enabled ) {
			/* force the option to be no */
			$ap_options['enabled'] = 'no';
			update_option( 'yith_ppwc_ap_gateway_options', $ap_options );
			$class = 'disabled';
		}

		/* Domain Registration */
		$apay_option_description .= '<div class="yith-ppwc-options-apple-pay-checks-container ' . $class . '">';
		$apay_option_description .= '<div class="yith-ppwc-options-apple-pay-checks domain-registration">';

		if ( empty( $ap_options['is_validated'] ) || ( isset( $ap_options['is_validated'] ) && ! $ap_options['is_validated'] ) ) {
			$link = 'https://www.paypal.com/uccservicing/apm/applepay';
			if ( 'sandbox' === YITH_PayPal::get_instance()->get_gateway()->get_environment() ) {
				$link = 'https://www.sandbox.paypal.com/uccservicing/apm/applepay';
			}

			$apay_option_description .= '<p><strong class="status_error">'. esc_html__( 'Note:', 'yith-paypal-payments-for-woocommerce' ) . '</strong>';
			//translators: placeholders are html tags
			$apay_option_description .= sprintf( esc_html__( 'To ensure payments work, you must register the domain. %1$sClick here to register the domain now >%2$s', 'yith-paypal-payments-for-woocommerce' ),
			'<a href="' . $link . '" target="_blank">',
			'</a>'
			);
			$apay_option_description .= '<br>' . esc_html__( 'After registering the domain, it will be validated on the first payment', 'yith-paypal-payments-for-woocommerce');
			$apay_option_description .= '</p>';
		}
		$apay_option_description .= '</div>';

		/* Browser/Device eligibility */
		$apay_option_description .= '<div class="yith-ppwc-options-apple-pay-checks device-eligibility">';
		if ( ! $this->is_browser_supported() ) {
			$supported_browsers       = implode( ', ', self::ALLOWED_USER_BROWSERS );
			$supported_devices        = implode( ', ', self::ALLOWED_USER_DEVICES );
			$apay_option_description .= '<span class="status_error">&#x2717;</span>';
			$apay_option_description .= esc_html_x( 'Your browser or device does not seem to be supported.', 'Admin Settings', 'yith-paypal-payments-for-woocommerce' );
			$apay_option_description .= '<br />' . esc_html_x( 'Supported Browsers: ', 'Admin Settings', 'yith-paypal-payments-for-woocommerce' ) . $supported_browsers;
			$apay_option_description .= '<br />' . esc_html_x( 'Supported Devices: ', 'Admin Settings', 'yith-paypal-payments-for-woocommerce' ) . $supported_devices;
		}
		$apay_option_description .= '</div></div>';

		$apple_pay = array(
			array(
				'title' => esc_html_x( 'Apple Pay Options', 'Admin option section title', 'yith-paypal-payments-for-woocommerce' ),
				'type'  => 'title',
				'desc'  => '',
				'id'    => 'yith_apple_pay_options',
			),

			array(
				'id'        => 'yith_ppwc_ap_gateway_options[enabled]',
				'title'     => esc_html_x( 'Enable Apple Pay payment', 'Admin title option', 'yith-paypal-payments-for-woocommerce' ),
				'desc'      => $apay_option_description,
				'type'      => 'yith-field',
				'yith-type' => 'onoff',
				'default'   => 'no',
				'class'     => ! $can_apple_pay || ! $this->paypal_enabled ? 'disabled' : '',
			),

			array(
				'type' => 'sectionend',
				'id'   => 'yith_apple_pay_options_end',
			),
		);

		return array_merge( $settings, $apple_pay );
	}

	/**
	 * Add Apple Pay button style settings to related tab settings
	 *
	 * @param array $settings array of options.
	 * @return array $settings list of settings.
	 */
	public function add_apay_style_settings( $settings ) {
		/* Apple Pay Options */
		$can_apple_pay = $this->can_apple_pay();
		$gp_options    = get_option( 'yith_ppwc_ap_gateway_options', array() );

		if ( ! $can_apple_pay || 'no' === $gp_options['enabled'] ) {
			return $settings;
		}

		$apple_pay = array(
			array(
				'title' => esc_html_x( 'Apple Pay Button', 'Admin option section title', 'yith-paypal-payments-for-woocommerce' ),
				'type'  => 'title',
				'desc'  => '',
				'id'    => 'yith_apple_pay_options',
			),

			array(
				'id'        => 'yith_ppwc_ap_gateway_options[buttonType]',
				'title'     => esc_html_x( 'Button Type', 'Admin title option', 'yith-paypal-payments-for-woocommerce' ),
				'desc'      => esc_html_x( 'Select the type of the button based on Apple types', 'Admin description option', 'yith-paypal-payments-for-woocommerce' ),
				'type'      => 'yith-field',
				'yith-type' => 'select',
				'options'   => array(
					'book'     => esc_html_x( 'Book', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'buy'      => esc_html_x( 'Buy', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'checkout' => esc_html_x( 'Checkout', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'donate'   => esc_html_x( 'Donate', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'order'    => esc_html_x( 'Order', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'pay'      => esc_html_x( 'Pay', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'plain'    => esc_html_x( 'Plain', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
				),
				'default'   => 'pay',
			),

			array(
				'id'        => 'yith_ppwc_ap_gateway_options[buttonColor]',
				'title'     => esc_html_x( 'Button Color', 'Admin title option', 'yith-paypal-payments-for-woocommerce' ),
				// translators:placeholders are html tags.
				'desc'      => sprintf( esc_html_x( 'Choose the ApplePay button color. The recommended color is %1$sblack%2$s.', 'Admin option, the placeholder are tags', 'yith-paypal-payments-for-woocommerce' ), '<strong>', '</strong>' ),
				'type'      => 'yith-field',
				'yith-type' => 'select-images',
				'options'   => array(
					'black-rect'         => array(
						'label' => esc_html_x( 'Black', 'Option: Button color', 'yith-paypal-payments-for-woocommerce' ),
						'image' => YITH_PAYPAL_PAYMENTS_URL . 'assets/images/applepay_black.png',
					),
					'white-rect'         => array(
						'label' => esc_html_x( 'White', 'Option: Button color', 'yith-paypal-payments-for-woocommerce' ),
						'image' => YITH_PAYPAL_PAYMENTS_URL . 'assets/images/applepay_white.png',
					),
					'white-outline-rect' => array(
						'label' => esc_html_x( 'White with outline', 'Option: Button color', 'yith-paypal-payments-for-woocommerce' ),
						'image' => YITH_PAYPAL_PAYMENTS_URL . 'assets/images/applepay_white_outline.png',
					),
				),
				'default'   => 'black-rect',
			),

			array(
				'id'        => 'yith_ppwc_ap_gateway_options[buttonLocale]',
				'title'     => esc_html_x( 'Button Language', 'Admin title option', 'yith-paypal-payments-for-woocommerce' ),
				'desc'      => esc_html_x( 'Set the button language. Browser means that broswer country and language is set.', 'Admin description option', 'yith-paypal-payments-for-woocommerce' ),
				'type'      => 'yith-field',
				'yith-type' => 'select',
				'options'   => array(
					'browser' => esc_html_x( 'Browser Language', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'ar-AB'   => esc_html_x( 'Arabic', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'ca-ES'   => esc_html_x( 'Catalan', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'cs-CZ'   => esc_html_x( 'Czech', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'da-DK'   => esc_html_x( 'Danish', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'de-DE'   => esc_html_x( 'German', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'el-GR'   => esc_html_x( 'Greek', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'en-AU'   => esc_html_x( 'English (Australia)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'en-GB'   => esc_html_x( 'English (United Kingdom)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'en-US'   => esc_html_x( 'English (United States)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'es-ES'   => esc_html_x( 'Spanish (Spain)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'es-MX'   => esc_html_x( 'Spanish (Mexico)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'fi-FI'   => esc_html_x( 'Finnish', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'fr-CA'   => esc_html_x( 'French (Canada)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'fr-FR'   => esc_html_x( 'French (France)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'he-IL'   => esc_html_x( 'Hebrew', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'hi-IN'   => esc_html_x( 'Hindi', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'hr-HR'   => esc_html_x( 'Croatian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'hu-HU'   => esc_html_x( 'Hungarian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'id-ID'   => esc_html_x( 'Indonesian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'it-IT'   => esc_html_x( 'Italian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'ja-JP'   => esc_html_x( 'Japanese', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'ko-KR'   => esc_html_x( 'Korean', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'ms-MY'   => esc_html_x( 'Malay', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'nb-NO'   => esc_html_x( 'Norwegian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'nl-NL'   => esc_html_x( 'Dutch', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'pl-PL'   => esc_html_x( 'Polish', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'pt-BR'   => esc_html_x( 'Portuguese (Brazil)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'pt-PT'   => esc_html_x( 'Portuguese (Portugal)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'ro-RO'   => esc_html_x( 'Romanian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'ru-RU'   => esc_html_x( 'Russian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'sk-SK'   => esc_html_x( 'Slovak', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'sv-SE'   => esc_html_x( 'Swedish', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'th-TH'   => esc_html_x( 'Thai', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'tr-TR'   => esc_html_x( 'Turkish', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'uk-UA'   => esc_html_x( 'Ukrainian', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'vi-VN'   => esc_html_x( 'Vietnamese', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'zh-CN'   => esc_html_x( 'Chinese (Simplified)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'zh-HK'   => esc_html_x( 'Chinese (Hong Kong)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
					'zh-TW'   => esc_html_x( 'Chinese (Traditional)', 'Admin option value label', 'yith-paypal-payments-for-woocommerce' ),
				),
				'default'   => 'browser',
			),

			array(
				'type' => 'sectionend',
				'id'   => 'yith_apple_pay_options_end',
			),
		);

		$settings['customization'] = array_merge( $settings['customization'], $apple_pay );

		return $settings;
	}

	/**
	 * Get apple pay activation url
	 *
	 * @since 2.22.0
	 * @return string
	 */
	public function get_apple_pay_activation_url() {
		if ( 'sandbox' === $this->environment ) {
			$url = 'https://www.sandbox.paypal.com/bizsignup/add-product?product=payment_methods&capabilities=APPLE_PAY';
		} else {
			$url = 'https://www.paypal.com/bizsignup/add-product?product=payment_methods&capabilities=APPLE_PAY';
		}

		return $url;
	}


	/**
	 * Returns single instance of the class
	 *
	 * @return YITH_PayPal_Apple_Pay
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Returns apple sdk url
	 *
	 * @return string
	 */
	public static function get_sdk_url() {
		return self::$sdk_url;
	}
	/**
	 * Returns if we are in google pay payment process
	 *
	 * @return string
	 */
	public static function is_applepay_flow() {
		$payed_with_applepay = WC()->session->get( 'yith_ppwc_funding_source' );
		if ( self::get_funding_source() === $payed_with_applepay ) {
			return true;
		}
		return false;
	}
	/**
	 * Check if product need shipping.
	 *
	 * @param WC_Product $product woocommerce product.
	 * @return bool
	 */
	public function check_if_need_shipping( $product ) {
		if (
			! wc_shipping_enabled()
			|| 0 === wc_get_shipping_method_count(
				true
			)
		) {
			return false;
		}
		$needs_shipping = false;

		if ( $product->needs_shipping() ) {
			$needs_shipping = true;
		}

		return $needs_shipping;
	}

	/** 
	 * Validare or not a merchant
	 *
	 * @param bool $valid .
	 */
	public static function validate_merchant( $valid ) {
		$ap_options = get_option( 'yith_ppwc_ap_gateway_options', array() );
		
		if ( $valid ) {
			$ap_options['is_validated'] = true;
		} else {
			$ap_options['is_validated'] = false;
		}

		update_option( 'yith_ppwc_ap_gateway_options', $ap_options );
	}

	/**
	 * Returns localized args for apple pay js
	 *
	 * @return array $data array of args.
	 */
	public function get_script_localize_args() {
		$settings       = get_option( 'yith_ppwc_ap_gateway_options' );
		$environment    = YITH_PayPal::get_instance()->get_gateway()->get_environment();
		$context        = '';
		$order_id       = '';
		$product_data   = '';
		$needs_shipping = false;

		if ( is_product() ) {
			$context = 'product';
			$product = wc_get_product( get_the_id() );
			if ( $product ) {
				$product_data = array(
					'product' => array(
						'product_id'    => get_the_id(),
						'needShipping'  => $this->check_if_need_shipping( $product ),
						'product_stock' => $product->get_stock_status(),
					),
				);

				$needs_shipping = $this->check_if_need_shipping( $product );
			}
		} elseif ( is_cart() ) {
			$context = 'cart';
			$needs_shipping = WC()->cart->needs_shipping();
		} elseif ( is_checkout() ) {
			$context = 'checkout';
			$needs_shipping = WC()->cart->needs_shipping();
		}

		$data = array(
			'environment'   => $environment,
			'context'       => $context,
			'buttonColor'   => isset( $settings['buttonColor'] ) ? str_replace( '-rect', '', $settings['buttonColor'] ) : 'black',
			'buttonType'    => isset( $settings['buttonType'] ) ? $settings['buttonType'] : 'pay',
			'buttonLocale'  => isset( $settings['buttonLocale'] ) ? $settings['buttonLocale'] : 'pay',
			'orderId'       => is_wc_endpoint_url( 'order-pay' ) ? $order_id : '',
			'fundingSource' => $this::get_funding_source(),
			'needShipping'  => $needs_shipping,
		);

		if ( $product_data ) {
			$data = array_merge( $data, $product_data );
		}

		return $data;
	}

	/**
	 * Return if apple pay is enabled
	 *
	 * @return bool
	 */
	public function is_enabled() {
		$enabled = $this->settings['enabled'];

		$is_https = isset( $_SERVER['HTTPS'] ) && 'on' === $_SERVER['HTTPS'];
		return $is_https && 'yes' === $enabled && $this->is_valid;
	}

	/**
	 * Enqueue the apple pay script
	 *
	 * @return void
	 */
	public function enqueue_scripts_style() {
		if ( is_checkout() && has_block( 'woocommerce/checkout-totals-block' ) || ( is_cart() && has_block( 'woocommerce/cart-totals-block' ) ) ) {
			/* blocks js are loaded by blocks integration class */
			wp_dequeue_script( 'yith-ppwc-applepay-sdk' );
			wp_dequeue_script( 'yith-ppwc-applepay' );
		}
	}

	/**
	 * Add apple pay button
	 *
	 * @return void
	 */
	public function add_button() {
		if ( $this->enabled && $this->is_browser_supported() ) {
			$funding_source = WC()->session->get( 'yith_ppwc_funding_source' );
			if ( is_cart() && $funding_source === $this->get_funding_source() ) {
				return;
			}

			if ( is_checkout() && has_block( 'woocommerce/checkout-totals-block' ) ) {
				// avoid to add container on checkout block.
				return;
			}
			$display = '';
			if ( is_product() ) {
				global $post;
				if ( ! empty( $post ) ) {
					$product = wc_get_product( $post->ID );
				}
				/* if product has no default variations set we hide the button - on reset or variations change it will be managed by js */
				if ( ! empty( $product ) && $product->is_type( 'variable' ) && empty( $product->get_default_attributes() ) ) {
					$display = 'display:none;';
				}
			}
			/**
			 * APPLY_FILTERS: yith_ppwc_applepay_button_custom_height
			 *
			 * Change the button height. Value is in px.
			 *
			 * @param int $value default is 50.
			 */
			$custom_height = 'height:' . apply_filters( 'yith_ppwc_applepay_button_custom_height', '50' ) . 'px; --apple-pay-button-height:' . apply_filters( 'yith_ppwc_applepay_button_custom_height', '50' ) . 'px';

			$w = isset( get_option( 'yith_ppwc_button_size' )['dimensions']['width'] ) ? get_option( 'yith_ppwc_button_size' )['dimensions']['width'] : '100';
			$u = isset( get_option( 'yith_ppwc_button_size' )['unit'] ) ? get_option( 'yith_ppwc_button_size' )['unit'] : 'percentage';
			$u = 'percentage' === $u ? '%' : 'px';
			/**
			 * APPLY_FILTERS: yith_ppwc_applepay_button_custom_width
			 *
			 * Change the button width. Value can be % or px.
			 *
			 * @param int $value default is the buttons width in settings.
			 */
			$custom_width = 'width:' . apply_filters( 'yith_ppwc_applepay_button_custom_width', $w . $u ) . ';';
			echo '<div id="applepay-container" style="' . esc_html( $display ) . esc_html( $custom_width ) . esc_html( $custom_height ) . '"></div>';
		}
	}

	/**
	 * Add the Apple Pay to sdk
	 *
	 * @param array $args List of args for the add_query_arg.
	 *
	 * @return array
	 */
	public function add_apple_pay_to_sdk( $args ) {
		$args['components'] = $args['components'] . ',applepay';
		return $args;
	}

	/**
	 * Get Apple Pay funding source slug
	 *
	 * @return string
	 */
	public static function get_funding_source() {
		return 'yith_ppwc_apple_pay';
	}

	/**
	 * Get Apple Pay funding source label (default: Apple Pay)
	 *
	 * @return string
	 */
	public static function get_funding_source_label() {
		/**
		* APPLY_FILTERS: yith_ppwc_apple_play_funding_label
		*
		* Set the apple pay funding title label.
		*
		* @param string $label (Apple Pay is default);
		*/
		return esc_html( apply_filters( 'yith_ppwc_apple_play_funding_label', __( 'Apple Pay (via PayPal)', 'yith-paypal-payments-for-woocommerce' ) ) );
	}

	/**
	 * Provide the matrix of supported country and currency matrix
	 *
	 * @return array $list .
	 */
	public function get_supported_country_currency() {
		return apply_filters(
			'yith_ppwc_applepay_supported_country_currency',
			array(
				'AU' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'AT' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'BE' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'BG' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'CA' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'CY' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'CZ' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'DK' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'EE' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'FI' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'FR' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'DE' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'GR' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'HU' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'IE' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'IT' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'LV' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'LI' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'LT' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'LU' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'MT' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'NO' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'NL' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'PL' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'PT' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'RO' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'SK' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'SI' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'ES' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'SE' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'GB' => array(
					'AUD',
					'BRL',
					'CAD',
					'CHF',
					'CZK',
					'DKK',
					'EUR',
					'GBP',
					'HKD',
					'HUF',
					'ILS',
					'JPY',
					'MXN',
					'NOK',
					'NZD',
					'PHP',
					'PLN',
					'SEK',
					'SGD',
					'THB',
					'TWD',
					'USD',
				),
				'US' => array(
					'AUD',
					'CAD',
					'EUR',
					'GBP',
					'JPY',
					'USD',
				),
			)
		);
	}
}
